/***************************************************************************
 *
 * Copyright (c) 2013 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#ifndef SUBDIVISION_H_INCLUDED
#define SUBDIVISION_H_INCLUDED

#include <sys/types.h>

#include <fstream>
#include <string>

#include "CalibrationTypes.h"

namespace LayerManagerCalibration
{
    class Subdivision
    {
    public:
        enum Type
        {
            TOUCH,
            BUTTON,
            SLIDER
        };

        bool parseStream(std::ifstream& stream);

        Type getType() const;
        const std::string& getName() const;
        uint getId() const;
        const coordinate getTopLeftCoordinate() const;
        const coordinate getBottomRightCoordinate() const;
        uint getWidth() const;
        uint getHeight() const;
        virtual bool validate();
        void clipCoordinate(const coordinate& coordIn, coordinate& coordOut);
        bool getUncalibrated() const;

    protected:
        Subdivision(Type type);
        virtual ~Subdivision();
        virtual bool handleUnknownKey(const std::string& key, const std::string& value) = 0;

    private:
        Type m_eType;
        std::string m_sName;
        uint m_u32Id;
        uint m_u32TopLeftCoordinateX;
        uint m_u32TopLeftCoordinateY;
        uint m_u32BottomRightCoordinateX;
        uint m_u32BottomRightCoordinateY;
        bool m_bUncalibrated;

        bool m_bNameParsed;
        bool m_bIdParsed;
        bool m_bTopLeftXParsed;
        bool m_bTopLeftYParsed;
        bool m_bBottomRightXParsed;
        bool m_bBottomRightYParsed;
    };

    inline Subdivision::Subdivision(Subdivision::Type type)
        : m_eType(type),
          m_bUncalibrated(false),
          m_bNameParsed(false),
          m_bIdParsed(false),
          m_bTopLeftXParsed(false),
          m_bTopLeftYParsed(false),
          m_bBottomRightXParsed(false),
          m_bBottomRightYParsed(false)
    {
    }

    inline Subdivision::~Subdivision()
    {
    }

    inline Subdivision::Type Subdivision::getType() const
    {
        return m_eType;
    }

    inline const std::string& Subdivision::getName() const
    {
        return m_sName;
    }

    inline uint Subdivision::getId() const
    {
        return m_u32Id;
    }

    inline const coordinate Subdivision::getTopLeftCoordinate() const
    {
        coordinate coord = {m_u32TopLeftCoordinateX,
                            m_u32TopLeftCoordinateY};
        return coord;
    }

    inline const coordinate Subdivision::getBottomRightCoordinate() const
    {
        coordinate coord = {m_u32BottomRightCoordinateX,
                            m_u32BottomRightCoordinateY};
        return coord;
    }

    inline uint Subdivision::getWidth() const
    {
        return m_u32BottomRightCoordinateX - m_u32TopLeftCoordinateX;
    }

    inline uint Subdivision::getHeight() const
    {
        return m_u32BottomRightCoordinateY - m_u32TopLeftCoordinateY;
    }

    inline bool Subdivision::getUncalibrated() const
    {
        return m_bUncalibrated;
    }
}

#endif
